import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Camera, User, Calendar, Star, Sparkles, TrendingUp, Award } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MySessions.css'

const MySessions = () => {
  const [sessions, setSessions] = useState([])

  useEffect(() => {
    loadSessions()
    const interval = setInterval(loadSessions, 500)
    return () => clearInterval(interval)
  }, [])

  const loadSessions = () => {
    const saved = localStorage.getItem('photoSessions')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setSessions(parsed)
      } catch (e) {
        setSessions([])
      }
    } else {
      const defaultSessions = [
        {
          id: 1,
          photographer: 'Анна Петрова',
          theme: 'Love Story',
          date: '2025-03-15',
          rating: 5
        },
        {
          id: 2,
          photographer: 'Михаил Соколов',
          theme: 'Портрет',
          date: '2025-03-10',
          rating: 4
        },
        {
          id: 3,
          photographer: 'Елена Волкова',
          theme: 'Семейная',
          date: '2025-03-05',
          rating: 5
        }
      ]
      setSessions(defaultSessions)
      localStorage.setItem('photoSessions', JSON.stringify(defaultSessions))
    }
  }

  const getThemeColor = (theme) => {
    const colors = {
      'Портрет': '#FF6B9D',
      'Семейная': '#C44569',
      'Love Story': '#FFB6C1',
      'Бренд': '#9B59B6'
    }
    return colors[theme] || '#FF6B9D'
  }

  const renderStars = (rating) => {
    return Array.from({ length: 5 }, (_, i) => (
      <Star
        key={i}
        size={20}
        fill={i < rating ? '#F59E0B' : 'none'}
        stroke={i < rating ? '#F59E0B' : '#E5E7EB'}
        strokeWidth={2}
      />
    ))
  }

  const averageRating = sessions.length > 0 
    ? (sessions.reduce((sum, s) => sum + s.rating, 0) / sessions.length).toFixed(1)
    : 0

  const totalSessions = sessions.length

  return (
    <div className="my-sessions-page">
      <div className="container">
        <motion.section 
          className="hero-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="hero-icon-wrapper"
            initial={{ scale: 0, rotate: -180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
          >
            <Sparkles size={80} strokeWidth={2} className="hero-icon" />
          </motion.div>
          <h1 className="hero-title">Мои съёмки</h1>
          <p className="hero-description">
            История ваших фотосессий и встреч с профессиональными фотографами
          </p>
        </motion.section>

        <motion.div 
          className="stats-grid"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <motion.div 
            className="stat-card"
            whileHover={{ scale: 1.05, rotate: 2 }}
            transition={{ type: "spring", stiffness: 300, damping: 10 }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(255, 107, 157, 0.15)' }}>
              <Camera size={36} strokeWidth={2.5} color="#FF6B9D" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{totalSessions}</div>
              <div className="stat-label">Съёмок</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ scale: 1.05, rotate: -2 }}
            transition={{ type: "spring", stiffness: 300, damping: 10 }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(196, 69, 105, 0.15)' }}>
              <User size={36} strokeWidth={2.5} color="#C44569" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{new Set(sessions.map(s => s.photographer)).size}</div>
              <div className="stat-label">Фотографов</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ scale: 1.05, rotate: 2 }}
            transition={{ type: "spring", stiffness: 300, damping: 10 }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(245, 158, 11, 0.15)' }}>
              <Award size={36} strokeWidth={2.5} color="#F59E0B" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{averageRating}</div>
              <div className="stat-label">Средняя оценка</div>
            </div>
          </motion.div>
        </motion.div>

        <ImageSlider images={[
          '/images/photo-1.jpg',
          '/images/photo-2.jpg',
          '/images/photo-3.jpg',
          '/images/photo-4.jpg'
        ]} />

        <motion.section 
          className="sessions-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title">Ваши фотосессии</h2>
          {sessions.length > 0 ? (
            <div className="sessions-grid">
              {sessions.map((session, index) => (
                <motion.div 
                  key={session.id} 
                  className="session-card"
                  initial={{ opacity: 0, scale: 0.9, rotate: -5 }}
                  animate={{ opacity: 1, scale: 1, rotate: 0 }}
                  transition={{ delay: index * 0.15, duration: 0.5 }}
                  whileHover={{ 
                    scale: 1.05, 
                    rotate: 3,
                    boxShadow: `0 20px 40px ${getThemeColor(session.theme)}30`
                  }}
                  style={{ 
                    borderTop: `4px solid ${getThemeColor(session.theme)}`,
                    background: `linear-gradient(135deg, rgba(255, 255, 255, 0.98) 0%, rgba(255, 245, 247, 0.95) 100%)`
                  }}
                >
                  <div className="session-header">
                    <div className="session-theme-badge" style={{ background: `${getThemeColor(session.theme)}20`, color: getThemeColor(session.theme) }}>
                      {session.theme}
                    </div>
                    <div className="session-rating">
                      {renderStars(session.rating)}
                    </div>
                  </div>
                  <div className="session-content">
                    <div className="session-info-item">
                      <User size={20} strokeWidth={2} color={getThemeColor(session.theme)} />
                      <span className="session-info-text">{session.photographer}</span>
                    </div>
                    <div className="session-info-item">
                      <Calendar size={20} strokeWidth={2} color={getThemeColor(session.theme)} />
                      <span className="session-info-text">
                        {new Date(session.date).toLocaleDateString('ru-RU', { 
                          day: 'numeric', 
                          month: 'long', 
                          year: 'numeric' 
                        })}
                      </span>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <div className="empty-icon-circle">
                <Camera size={64} strokeWidth={1.5} />
              </div>
              <h3 className="empty-title">Нет запланированных съёмок</h3>
              <p className="empty-text">Запишитесь на фотосессию прямо сейчас</p>
            </div>
          )}
        </motion.section>

        <motion.section 
          className="info-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.6 }}
        >
          <div className="info-card">
            <h2 className="info-title">Как проходит фотосессия</h2>
            <div className="info-steps">
              <div className="info-step">
                <div className="step-number">1</div>
                <div className="step-content">
                  <h3 className="step-title">Запись</h3>
                  <p className="step-text">Выберите стиль съёмки и удобную дату</p>
                </div>
              </div>
              <div className="info-step">
                <div className="step-number">2</div>
                <div className="step-content">
                  <h3 className="step-title">Подготовка</h3>
                  <p className="step-text">Обсуждение деталей с фотографом</p>
                </div>
              </div>
              <div className="info-step">
                <div className="step-number">3</div>
                <div className="step-content">
                  <h3 className="step-title">Съёмка</h3>
                  <p className="step-text">Профессиональная фотосессия</p>
                </div>
              </div>
              <div className="info-step">
                <div className="step-number">4</div>
                <div className="step-content">
                  <h3 className="step-title">Результат</h3>
                  <p className="step-text">Готовые фото через 3-5 дней</p>
                </div>
              </div>
            </div>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default MySessions

